/**********************************************************************************************
* Device      : AD18F08
* Author      : Sunwangzhi
* Version     : V0.0.1
* Date        : 2022.3.24
* Description : CCP的比较功能选择；根据不同的比较模式，可选择查看PB5输出或ADC中断
                若测试COM_MODE2或COM_MODE3中断触发，请注释串口打印功能，否则会影响触发现象;
                查看ADC,可开放Uart打印功能
***********************************************************************************************/

#include "AD18F08.h"
#include <stdio.h>
#include "usart.h"


/*********************************************************************************************
* 常用配置字功能说明：
*
* _FINTOSC_DIV1     :  内部时钟分频选择
* _DBG_OFF          :  禁用DEBUG模式（使能后DEBUG 口不能做普通IO用）
* _LVREN_OFF         :  LVR使能位
* _LVT24V           :  选择LVR电压为2.4V
* _FLASH_CP1_OFF    :  1K~2K*16 FLASH保护位
* _FLASH_CP0_OFF    :  0~1K*16 FLASH保护位
* _IDIS_OFF	     	:  模拟IO打开，数据读取控制位
* _CKSUM_OFF        :  使能程序校验
* _CBP_OFF			:  代码保护选择位
* _RESETE_OFF       :  禁用外部复位
* _RSEL				:  端口上下拉电阻选择
* _STV_ON		    :  堆栈溢出复位使能位
* _PSUT_2		    :  上电复位时间选择位
* _FCPU_2T          :  选择指令周期为2T模式
* _WDTE_OFF        :  看门狗使能位
* _TWDT_000         :  看门狗溢出时间选择位
* _OSCM_HIRC_16M    :  选择内部高速时钟16MHz
* _ADCSP_ON			:  ADC采集速度选择位

*
* 注意：1.配置字若无特别需求请勿修改；
*       2.其他配置字及相应功能见Datasheet及头文件；
*
************************************************************************************************/
//CCP的比较功能
__CONFIG(1,_FINTOSC_DIV1_1L     & _DBG_OFF_1L       & _LVREN_OFF_1L     & _LVR24V_1L                                                  );
__CONFIG(2,_FLASH_CP1_OFF_2L    & _FLASH_CP0_OFF_2L & _IDIS_OFF_2L     & _CKSUM_OFF_2L  & _CBP_OFF_2L   & _RESETE_OFF_2L & _RSEL_LOW_2L & _STV_ON_2L  );
__CONFIG(3,_PSUT_2_3L           & _FCPU_2T_3L       & _WDTE_OFF_3L    & _TWDT_100_3L                                                );
__CONFIG(4,_OSCM_HIRC_16M_4L    & _MTPDIS_OFF_4L    &_ADCSP_ON_4L                                                                                     );

#define COM_MODE0    4       //比较模式，选择CCPR1匹配时将输出置为高电平（CCPIF位置1），输出引脚为PB5；TIMER1溢出时，PB5为低电平；
#define COM_MODE1    5       //比较模式，选择CCPR1匹配时将输出置为低电平（CCPIF位置1），输出引脚为PB5；TIMER1溢出时，PB5为高电平；
#define COM_MODE2    6       //比较模式，选择CCPR1匹配时将产生软件中断（CCPIF位置1，而PB5引脚不受影响）；并启动ADC采集（如果ADON=1）
#define COM_MODE3    7       //比较模式，选择CCPR1触发特殊事件（CCPIF位置1，PB5引脚不受影响）；CCP1清零TIMER1; 并启动ADC采集（如果ADON=1）

void GPIO_Init(void);
void ADC_init(void);
void Timer1_Com_Init(unsigned char com_mode);
//==============================================================================================
// ----Function: 主程序
// --Parameters: None
// -----Returns: None
//---------Note: 配置CCP比较模式
//==============================================================================================
bit Adc_Succee;
unsigned short Adc_Val;
void main (void)
{
    Adc_Succee = 0;
    Adc_Val = 0;
	GPIO_Init();
    ADC_init();
    usart_init();
	IPEN = 0;	   	//禁止中断优先级
	PEIE = 1;    	//外设GIEL
	GIE  = 1;     	//全局GIEH

	ADIE=1;
	ADIF=0;
	ADIP=1;


	//Timer1_Com_Init(COM_MODE0);   	//选择CCPR1匹配时将输出置为高电平（CCPIF位置1），输出引脚为PB5；TIMER1溢出时，PB5为低电平；


	//Timer1_Com_Init(COM_MODE1); 	    //选择CCPR1匹配时将输出置为低电平（CCPIF位置1），输出引脚为PB5；TIMER1溢出时，PB5为高电平；

	Timer1_Com_Init(COM_MODE2);     //选择CCPR1匹配时将产生软件中断（CCPIF位置1，而PB5引脚不受影响）；并启动ADC采集（如果ADON=1）
	  //ADON=1; 	//使能ADC

	//	Timer1_Com_Init(COM_MODE3);		//选择CCPR1触发特殊事件（CCPIF位置1，PB5引脚不受影响）；CCP1清零TIMER1; 并启动ADC采集（如果ADON=1）
	//  ADON=1;		//使能ADC


	TMR1H=0x0f;
	TMR1L=0xFF;

	CCPR1H=0xfF;
	CCPR1L=0xff;
	T1CON0=0x01;   //使能tmr1  选Fcpu做时钟
	while(1)
    {
        if(Adc_Succee == 1)
        {
            Adc_Succee = 0;
            PB2 =~PB2;                                  //PB2与PB3比较现象，查看功能正确性
            Adc_Val = (ADRESH << 8) | ADRESL;           //从ADC中读取ADC值
            printf("ADC_VAL =  %d\r\n",Adc_Val);        //打印ADC值，波特率为19200； 模拟串口为PA4
        }
    }

}

//高优先级中断服务函数

void interrupt ISR_h (void)
{
//进入中断后系统自动关闭中断
   if(ADIF == 1 && ADIE == 1)
   {
		ADIF=0;		//ADC中断标志位清0
        Adc_Succee = 1;
		PB3=~PB3;
   }
  	if(CCPIF == 1  &&  CCPIE == 1)
	{
		CCPIF = 0;	//CCP中断标志位清0
	    PB5 =~PB5;
	}
}

//低优先级中断服务函数
void interrupt low_priority ISR_l (void)
{
//进入中断后系统自动关闭中断

}

/*******************************************************************************
// ----Function: GPIO_Init
// --Parameters:
// -----Returns: None
//---------Note:所有IO初始化
*******************************************************************************/

void GPIO_Init(void)
{
	TRISA = 0X00;   //PA输出模式
	TRISB = 0X00;   //PB输出模式

	PORTA = 0X00;	//PORTA输出0
	PORTB = 0X00;	//PORTB输出0

	PAPD = 0XFF;    //关闭PA下拉
	PAPU = 0XFF;    //关闭PA上拉
	PBPD = 0XFF;    //关闭PB下拉
	PBPU = 0XFF;    //关闭PB上拉

	PAOD = 0X00;    //关PA开漏输出
	PBOD = 0X00;    //关PB开漏输出

	ANASEL0 = 0X00; //关PA模拟量通道
	ANASEL1 = 0X00; //关PB模拟量通道
}
/*******************************************************************************
// ----Function: Timer1_Com_Init
// --Parameters:
// -----Returns: None
//---------Note:CCP 比较模式配置
*******************************************************************************/
void Timer1_Com_Init(unsigned char com_mode)
{
	TRISB5=0;               //PB5为输出模式
	CCPCON |= com_mode ;    //比较模式
	CCPIF = 0;
	CCPIE = 1;
	CCPIP = 1;
}
/*******************************************************************************
// ----Function: ADC_init
// --Parameters:
// -----Returns: None
//---------Note: 使能ADC
*******************************************************************************/
void ADC_init(void)
{
    TRISA5 = 1;PA5PD = 1;PA5PU = 1;PA5OD = 0;PA5AEN = 1;

    ADCON0 = 0X09;  //配置PA5为ADC采集通道，使能A/D模拟使能位；ADCMPMODE=0时采集值大于ADCMP1H产生中断ADCMP1IF或者小于ADCMP0H产生中断ADCMP0IF
	ADCON1 = 0XBE;  //参考电压选择VDD，A/D转换时钟选择FOSC/64(ADSP=1)

	ADIF=0;	  //清中断标志
	ADIP=1;   //高优先级中断
	ADIE=1;   //开AD中断
	ADON=1;   //打开A/D模拟使能位
}
